<?php

namespace App\Models;

use App\Enums\BLTypes;
use App\Enums\ShipmentTypes;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Concerns\HasUlids;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Facades\Auth;

class BL extends Model
{
    use HasFactory, SoftDeletes;
    use HasUlids;

    protected $casts = [
        'bl_type' => BLTypes::class,
        'shipment_type' => ShipmentTypes::class,
    ];
    
    /**
     * The "booted" method of the model.
     */
    protected static function booted(): void
    {
        // dd(Company::where('organization_id', auth()->user()->organization_id));
        static::addGlobalScope('organization', function (Builder $builder) {
            $builder->where('organization_id', auth()->user()->organization_id);
        });

        // static::addGlobalScope('office', function (Builder $builder) {
        //     if (auth()->user()->is_customer()) {
        //         return;
        //     }
            
        //     $user_office = OfficeUser::find(Auth::id())->office()->first();
        //     $user_office_id = $user_office?$user_office->id:null;
            
        //     $builder->where('office_id', $user_office_id);
        // });

        static::addGlobalScope('office', function (Builder $builder) {
            if (auth()->user()->is_customer()) {
                return;
            }
            
            $user_offices = OfficeUser::find(Auth::id())->office()->get();
            $user_office_ids = [];
            foreach ($user_offices as $uoffice) {
                $user_office_ids[] = $uoffice->id;
            }
            $builder->whereIn('office_id', $user_office_ids);
        });
    }

    public function uniqueIds()
    {
        // Tell Laravel you want to use ULID for your secondary 'ulid' column instead
        return [
            // 'ulid',
            'bl',
        ];
    }

    /**
     * Get the company that owns the Billoflading
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function invoice_to_company(): BelongsTo
    {
        return $this->belongsTo(Company::class, 'invoice_to_company_id', 'id');
    }

    /**
     * Get the shipper that owns the Billoflading
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function delivery_agent(): BelongsTo
    {
        return $this->belongsTo(Office::class, 'delivery_agent_id', 'id')->with(['country', 'city']);
    }

    /**
     * Get the shipper that owns the Billoflading
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function place_of_loading(): BelongsTo
    {
        return $this->belongsTo(Port::class, 'place_of_loading_id', 'id');
    }

    /**
     * Get the shipper that owns the Billoflading
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function port_of_discharge(): BelongsTo
    {
        return $this->belongsTo(Port::class, 'port_of_discharge_id', 'id');
    }

    /**
     * Get the shipper that owns the Billoflading
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function entry_by(): BelongsTo
    {
        return $this->belongsTo(User::class, 'entry_by_user_id', 'id');
    }

    /**
     * Get the shipper that owns the Billoflading
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function last_updated_by(): BelongsTo
    {
        return $this->belongsTo(User::class, 'last_updated_by_user_id', 'id');
    }

    /**
     * Get the shipper that owns the Billoflading
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function final_destination_city(): BelongsTo
    {
        return $this->belongsTo(City::class, 'final_destination_city_id', 'id');
    }

    /**
     * Get the shipper that owns the Billoflading
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function final_destination_country(): BelongsTo
    {
        return $this->belongsTo(Country::class, 'final_destination_country_id', 'id');
    }

    /**
     * Get all of the bl_containers for the Billoflading
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function bl_containers(): HasMany
    {
        return $this->hasMany(BLContainer::class, 'bl_id');
    }

    /**
     * Get all of the bl_containers for the Billoflading
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function trackings(): HasMany
    {
        return $this->hasMany(Tracking::class, 'bl_id');
    }

    /**
     * Get the shipper that owns the Billoflading
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function office(): BelongsTo
    {
        return $this->belongsTo(Office::class, 'office_id', 'id');
    }
}
