<?php

namespace App\Filament\Office\Resources;

use Filament\Forms;
use App\Models\User;
use Filament\Tables;
use Filament\Forms\Form;
use Filament\Tables\Table;
use Filament\Resources\Resource;
use Filament\Resources\Pages\Page;
use Illuminate\Support\Facades\Hash;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Office\Resources\UserResource\Pages;
use App\Filament\Office\Resources\UserResource\Pages\EditUser;
use App\Filament\Office\Resources\UserResource\Pages\CreateUser;

class UserResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-users';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('name')
                    ->required()
                    ->maxLength(255),
                Forms\Components\TextInput::make('email')
                    ->email()
                    ->required()
                    ->maxLength(255),
                // Forms\Components\DateTimePicker::make('email_verified_at'),
                Forms\Components\TextInput::make('password')
                    ->password()
                    ->maxLength(255)
                    ->dehydrateStateUsing(static fn (null|string $state): null|string => filled($state)?Hash::make($state):null)
                    ->required(static fn(Page $livewire): string => $livewire instanceof CreateUser)
                    ->dehydrated(static fn (null|string $state): bool => filled($state))
                    ->live(debounce: 500)
                    ->label(static fn(Page $livewire): string => ($livewire instanceof EditUser)?'New Password':'Password'),
                // Forms\Components\TextInput::make('profile_pic_path')
                //     ->maxLength(255),
                Forms\Components\FileUpload::make('profile_pic_path')
                    ->avatar(),
                Forms\Components\Toggle::make('is_admin'),
                // Forms\Components\TextInput::make('theme')
                //     ->maxLength(255)
                //     ->default('default'),
                // Forms\Components\TextInput::make('theme_color')
                //     ->maxLength(255),
                // Forms\Components\CheckboxList::make('roles')
                //     ->relationship('roles', 'name')
                //     ->columns(2)
                //     ->required(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\ImageColumn::make('profile_pic_path')
                    ->circular()
                    ->label('Photo'),
                Tables\Columns\TextColumn::make('name')
                    ->sortable()
                    ->searchable(),
                Tables\Columns\TextColumn::make('email')
                    ->searchable()
                    ->sortable(),

                Tables\Columns\ToggleColumn::make('email_verified_at')
                    ->beforeStateUpdated(function ($record, $state) {
                        // Runs before the state is saved to the database.
                        // return $state?date('Y-m-d H:i:s'):'0000-00-00 00:00:00';
                        return $state?date('Y-m-d H:i:s'):null;
                    })
                    ->updateStateUsing(function ($record, $state) {
                        return $record->update(['email_verified_at' => $state?date('Y-m-d H:i:s'):null]);
                    })
                    ->afterStateUpdated(function ($record, $state) {
                        // Runs after the state is saved to the database.
                    })
                    ->label('Verified'),
                // Tables\Columns\IconColumn::make('email_verified_at')
                //     ->label('Verified')
                //     ->boolean(),

                // Tables\Columns\ToggleColumn::make('is_admin')
                //     ->label('Admin'),
                // Tables\Columns\IconColumn::make('is_admin')
                //     ->label('Admin')
                //     ->boolean(),

                // Tables\Columns\TextColumn::make('deleted_at')
                //     ->dateTime()
                //     ->sortable(),
                // Tables\Columns\TextColumn::make('created_at')
                //     ->dateTime()
                //     ->sortable()
                //     ->toggleable(isToggledHiddenByDefault: true),
                // Tables\Columns\TextColumn::make('updated_at')
                //     ->dateTime()
                //     ->sortable()
                //     ->toggleable(isToggledHiddenByDefault: true),
                // Tables\Columns\TextColumn::make('theme')
                //     ->searchable(),
                // Tables\Columns\TextColumn::make('theme_color')
                //     ->searchable(),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
            // ->toggleColumnsTriggerAction(
            //     fn (Action $action) => $action
            //         ->button()
            //         ->label('Toggle columns'),
            // );
    }
    
    public static function getRelations(): array
    {
        return [
            //
        ];
    }
    
    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }    
    
    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
