<?php

namespace App\Http\Controllers;

use App\Models\BL;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Reposotories\BlReposotories;
use App\Filament\Office\Resources\BLResource;

class BLController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id, Request $request)
    {
        $bl = BL::where('bl', $id)->with([
            'port_of_discharge',
            'place_of_loading',
            'shipper',
            'consignee',
            'notify_party',
            'notify_party2',
            'delivery_agent',
            'bl_containers',
        ])->withoutGlobalScopes([
            // SoftDeletingScope::class,
            'organization',
            'office'
        ])->firstOrFail();
        
        $bl = (new BlReposotories($bl))->setDateFormat('d/m/Y')->formatBlData();
        $pdf_name  = $this->getPdfName($bl->bl_no);
        $bg = $request->has('pdf')||$request->has('email');

        if ($request->has('email')) {
            $pdf = Pdf::loadView('bls.pdf', compact(['data', 'bg']))->setPaper([0.0, 0.0, 598, 850], 'potrait')->setOption(['dpi' => 200]);
            // return $pdf->stream($pdf_name);
            // return view('bls.pdf', compact('data'));
            // return $pdf->save(storage_path('app/bls/'.$id).'/'.$pdf_name)->stream($pdf_name);
            return $pdf->download($pdf_name);
        }

        $pdf = $this->pdf($bl, $bg);
        return $pdf->stream($pdf_name);
    }

    function pdf($bl, $bg = false, $highRes = false) {
        // $view = $highRes?'bls.pdf':'bls.pdf-low-res';
        $view = $highRes?'bls.pdf':'bls.pdf-template';

        // $view = $bl->office_id==14?'bls.pdf-dhaka-office':$view;
        
        $dpi = $highRes?200:96;
        $bg_image = resource_path('images/bl/bl-front-low-res.jpg');

        return Pdf::loadView($view, compact(['bl', 'bg', 'bg_image']))->setPaper([0.0, 0.0, 598, 850], 'potrait')->setOption(['dpi' => $dpi]);
    }

    function getPdfName($bl_no = '0') : string {
        return 'bl-'.$bl_no.'-'.date('dmyHi').'.pdf';
    }

    public function certificate(string $id, Request $request)
    {
        $bl = BL::where('bl', $id)->with([
            'bl_certificates',
        ])->withoutGlobalScopes([
            // SoftDeletingScope::class,
            'organization',
            'office'
        ])->firstOrFail();
        
        if (count($bl->bl_certificates)<1) {
            return abort('403', 'This BL does not have any Certificate attached yet! Please contact to our office.');
        }
        
        $page_title = $bl->bl_no . '-' . Str::plural('Certificate', count($bl->bl_certificates));
        $contents = '';
        $page = 0;
        foreach ($bl->bl_certificates as $certificate) {
            if ($page != 0) {
                $contents .= "<div class='page-break'></div>";
            }
            $page++;
            $contents .= "<div class='page page".$page."'>" . $certificate->certificate . "</div>";
        }

        $pdf = Pdf::loadView('bls.certificates', compact(['page_title', 'contents']))->setPaper([0.0, 0.0, 598, 850], 'potrait');
        return $pdf->stream($page_title.'.pdf');
    }

    public function ridersheet(string $id, Request $request)
    {
        $bl = BL::where('bl', $id)->with([
            'bl_containers',
        ])->withoutGlobalScopes([
            // SoftDeletingScope::class,
            'organization',
            'office'
        ])->firstOrFail();
        
        if (!$bl->hasRiderSheet() || count($bl->bl_containers)<1) {
            return abort('403', 'This BL does not have any Rider Sheet attached yet! Please contact to our office.');
        }
        
        $page_title = $bl->bl_no . '-' . 'ridersheet';
        $logo = '/images/logos/sealloyd-logo-180x90.png';
        $registered_company_name = 'Sea Lloyd Shipping Lines Pte Ltd';

        $pdf = Pdf::loadView('bls.ridersheet', compact(['bl', 'logo', 'registered_company_name']))->setPaper([0.0, 0.0, 598, 850], 'potrait');
        return $pdf->stream($page_title.'.pdf');
    }

    public function getDateRangeFromMonth($month) : array {
        $months = [
            'this_month' => date('M Y'),
            'last_month' => date('M Y', strtotime('-1 month')),
            'last_two_months' => [
                date('M Y', strtotime('-1 month')),
                date('M Y')
            ],
            'last_three_months' => [
                date('M Y', strtotime('-2 month')),
                date('M Y')
            ],
            'last_six_months' => [
                date('M Y', strtotime('-5 month')),
                date('M Y')
            ],
        ];
        // dd($months);

        $first_date = date('Y-m-01', strtotime(
            is_array($months[$month])?
                $months[$month][0]:
                $months[$month])
        );

        $last_date = date('Y-m-t', strtotime(
            is_array($months[$month])?
                $months[$month][1]:
                $months[$month])
        );

        // dd($first_date, $last_date);

        // dd(date('Y-m-01', strtotime($months[request('month')])));
        // dd(date('Y-m-t', strtotime($months[request('month')])));

        return [$first_date, $last_date];
    }

    public function getDateRangeFromYear($year) : array {
        $years = [
            'this_year' => date('Y'),
            'last_year' => date('Y', strtotime('-1 year')),
            'last_two_years' => [
                date('Y', strtotime('-1 year')),
                date('Y')
            ],
            'last_three_years' => [
                date('Y', strtotime('-2 year')),
                date('Y')
            ],
            'last_six_years' => [
                date('Y', strtotime('-5 year')),
                date('Y')
            ],
        ];
        // dd($years);
        // dd($years[$year]);
        // dd(date('Y-m-d', strtotime('first day of January '.$years[$year])));
        // dd(gettype($years[$year].'-01-01'), gettype(date($years[$year].'-01-01')));

        $first_date = date(is_array($years[$year])?
                $years[$year][0]:
                $years[$year].'-01-01');

        $last_date = date((is_array($years[$year])?
                $years[$year][1]:
                $years[$year]).'-12-31');

        // dd($first_date, $last_date);

        // dd(date('Y-m-01', strtotime($years[request('year')])));
        // dd(date('Y-m-t', strtotime($years[request('year')])));

        return [$first_date, $last_date];
    }

    public function getDateFromDay($day): string {
        $date = [
            'today' => date('Y-m-d'),
            'yesterday' => date('Y-m-d', strtotime('-1 day')),
        ];
        return $date[$day];
    }

    public function getDateRange() : array|string|bool {
        $date_range = false;
        if (request('date_from') && request('date_until')) {
            $date_range = [
                request('date_from'),
                request('date_until')
            ];
        }
        if (request('month')) {
            $date_range = $this->getDateRangeFromMonth(request('month'));
        }
        if (request('year')) {
            $date_range = $this->getDateRangeFromYear(request('year'));
        }
        if (request('day')) {
            $date_range = $this->getDateFromDay(request('day'));
        }
        return $date_range;
    }

    public function report() {
        $date_range = $date_range = $this->getDateRange();

        $bls = BL::query()
            ->when(
                request('vessel'), 
                fn($query) => request('vessel_type')=='ocean'
                    ?($query->where('ocean_vessel_id', 'LIKE', "%".request('vessel')."%")->orWhere('ocean_vessel_old', 'LIKE', "%".request('vessel')."%"))
                    :($query->where('pre_carriage_vessel', 'LIKE', "%".request('vessel')."%"))
            )
            ->when(
                request('voyage'), 
                fn($query) => request('vessel_type')=='ocean'
                    ?($query->where('ocean_vessel_voy_id', 'LIKE', "%".request('vessel')."%")->orWhere('voy_no_destination_old', 'LIKE', "%".request('vessel')."%"))
                    :($query->where('voy_no_pre_carriage', 'LIKE', "%".request('vessel')."%"))
            )
            ->when(
                request('office_id'), 
                fn($query) => $query->where('office_id', request('office_id'))
            )
            ->when(
                request('place_of_loading_id'), 
                fn($query) => $query->where('place_of_loading_id', request('place_of_loading_id'))
            )
            ->when(
                request('port_of_discharge_id'), 
                fn($query) => $query->where('port_of_discharge_id', request('port_of_discharge_id'))
            )
            ->when(
                request('place_of_receipt'), 
                fn($query) => $query->where('place_of_receipt', 'LIKE', "%".request('place_of_receipt')."%")
            )
            ->when(
                request('final_destination'), 
                fn($query) => $query->where('final_destination', 'LIKE', "%".request('final_destination')."%")
            )
            ->when(
                request('bl_type'), 
                fn($query) => $query->whereIn('bl_type', request('bl_type'))
            )
            ->when(
                request('bl_status'), 
                fn($query) => $query->whereIn('bl_status', request('bl_status'))
            )
            ->when(
                request('is_final'), 
                // fn($query) => $query->where('is_final', true)
                fn($query) => $query->whereIsFinal(true)
            )
            ->when(
                request('locked'), 
                // fn($query) => $query->where('locked', true)
                fn($query) => $query->whereLocked(true)
            )
            ->when($date_range, function($query, $date_range) {
                if ((is_array($date_range) && isset($date_range[0]))) {
                    return $query->where(request('filter_by'), '>=', $date_range[0]);
                } else {
                    return $query->where(request('filter_by'), $date_range);
                }
            })
            ->when($date_range, function($query, $date_range) {
                if ((is_array($date_range) && isset($date_range[1]))) {
                    return $query->where(request('filter_by'), '<=', $date_range[1]);
                }
            })
            // ->when(
            //     request('date_from'), 
            //     fn($query) => $query->where(request('filter_by'), '>=', request('date_from'))
            // )
            // ->when(
            //     request('date_until'), 
            //     fn($query) => $query->where(request('filter_by'), '<=', request('date_until'))
            // )
            ->with([
                'port_of_discharge',
                'place_of_loading',
                'delivery_agent',
                'bl_containers',
                'office',
            ])
            ->withoutGlobalScopes([
                // SoftDeletingScope::class,
                'organization',
                // 'office'
            ])
            ->get()
            // ->dumpRawSql();
            ->map(function($bl) {
                // "pre_carriage_vessel" => "MT Jackeline New"
                // "voy_no_pre_carriage" => "53335ca2"
                $bl_data = [
                    'bl_no' => $bl->bl_no,
                    'bl_type' => $bl->bl_type->getLabel(),
                    'cargo_type' => $bl->cargo_type->getLabel(),
                    // 'loading_type' => $bl->loading_type->getLabel(),
                    'etd' => !empty($bl->etd)?Carbon::parse($bl->etd)->format('d M Y'):null,
                    'eta' => !empty($bl->eta)?Carbon::parse($bl->eta)->format('d M Y'):null,
                    'ocean_vessel' => empty($bl->ocean_vessel)?$bl->ocean_vessel_old:$bl->ocean_vessel->name,
                    'voy_no_destination' => empty($bl->ocean_vessel_voy)?$bl->voy_no_destination_old:$bl->ocean_vessel_voy->voy_no,
                    'place_of_loading' => empty($bl->place_of_loading_id)?$bl->place_of_loading_manual:($bl->place_of_loading->name .', '. $bl->place_of_loading->country_name),
                    'port_of_discharge' => empty($bl->port_of_discharge_id)?$bl->port_of_discharge_manual:($bl->port_of_discharge->name .', '. $bl->port_of_discharge->country_name),
                    'final_destination' => empty($bl->delivery_agent_id)?$bl->final_destination:($bl->delivery_agent->name .', '. $bl->delivery_agent->city->name.', '. $bl->delivery_agent->country->name),
                    'payable_at' => $bl->payable_at,
                    'place_issue' => $bl->place_issue,
                    'issue_date' => !empty($bl->issue_date)?Carbon::parse($bl->issue_date)->format('d M Y'):null,
                    'shipped_on_board' => !empty($bl->shipped_on_board)?Carbon::parse($bl->shipped_on_board)->format('d M Y'):null,
                    'locked' => $bl->locked,
                    'is_final' => $bl->is_final,
                    'customer_created' => $bl->customer_created,

                    'office' => !empty($bl->office_id)?$bl->office:null,

                    'created_at' => !empty($bl->created_at)?Carbon::parse($bl->created_at)->format('d M Y'):null,
                    'updated_at' => !empty($bl->updated_at)?Carbon::parse($bl->updated_at)->format('d M Y'):null,
                    'deleted_at' => !empty($bl->deleted_at)?Carbon::parse($bl->deleted_at)->format('d M Y'):null,
                    // 'containers' => collect($bl->bl_containers)->map(function($container) {
                    //     return [
                    //         'num' => BLResource::container_list($container),
                    //     ];
                    // }),
                    'bl_containers' => $bl->bl_containers,
                    'container_count' => count($bl->bl_containers),
                    'container_tues' => BLResource::tues($bl->bl_containers),
                ];

                return (object) $bl_data;
            });
        // dd($bls);
        // dd(OfficeUser::where('id', auth('admin')->user()->id)->with('office')->get());
        
        // return view('reports.bls', compact('bls', 'date_range'));
        return view('reports.bls')->with('bls', $bls)->with('date_range', $date_range);
        // $pdf = Pdf::loadView('reports.bls', compact(['bls', 'date_range']))->set_option('isRemoteEnabled', true);
        // return $pdf->stream('bl-report.pdf');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
