<?php

namespace App\Filament\Resources;

use App\Filament\Tables\BLTables;
use App\Models\BL;
use Filament\Tables;
use App\Enums\BLTypes;
use Filament\Forms\Form;
use Filament\Tables\Table;
use App\Filament\Forms\BLForms;
use Filament\Resources\Resource;
use Filament\Tables\Actions\Action;
use Filament\Tables\Filters\Filter;
use Illuminate\Database\Eloquent\Model;
use Filament\Forms\Components\DatePicker;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Enums\ActionsPosition;
use App\Filament\Resources\BLResource\Pages;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Resources\BLResource\RelationManagers;

class BLResource extends Resource
{
    protected static ?string $model = BL::class;

    protected static ?string $slug = 'shipping/bls';

    protected static ?string $navigationGroup = 'Shipping';
    protected static ?string $navigationIcon = 'heroicon-o-ticket';
    protected static ?string $navigationLabel = 'Bill Of Ladings';
    protected static ?string $modelLabel = 'Bill Of Lading';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                BLForms::makeBlLeftForm(true),
                BLForms::makeBlRightForm(true),
            ])
            ->columns(3);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns(
                BLTables::blTable(customer: true)
            )
            ->defaultSort('created_at', 'desc')
            ->searchOnBlur()
            ->filters([
                Filter::make('etd')
                    ->form([
                        DatePicker::make('etd_from')
                            ->label('ETD From (date)'),
                        DatePicker::make('etd_until')
                            ->label('ETD Until (date)'),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['etd_from'],
                                fn (Builder $query, $date): Builder => $query->whereDate('etd', '>=', $date),
                            )
                            ->when(
                                $data['etd_until'],
                                fn (Builder $query, $date): Builder => $query->whereDate('etd', '<=', $date),
                            );
                    }),
                
                // Filter::make('eta')
                //     ->form([
                //         DatePicker::make('eta_from')
                //             ->label('ETA From (date)'),
                //         DatePicker::make('eta_until')
                //             ->label('ETA Until (date)'),
                //     ])
                //     ->query(function (Builder $query, array $data): Builder {
                //         return $query
                //             ->when(
                //                 $data['eta_from'],
                //                 fn (Builder $query, $date): Builder => $query->whereDate('eta', '>=', $date),
                //             )
                //             ->when(
                //                 $data['eta_until'],
                //                 fn (Builder $query, $date): Builder => $query->whereDate('eta', '<=', $date),
                //             );
                //     }),
                Tables\Filters\SelectFilter::make('bl_type')
                    ->multiple()
                    ->options(BLTypes::class),
                Tables\Filters\SelectFilter::make('bl_status')
                    ->multiple()
                    ->options(fn (): array => \App\Models\Status::query()->pluck('status', 'id')->all()),
                Filter::make('is_final')
                    ->label('Only Final BLs')
                    ->toggle()
                    ->query(fn (Builder $query): Builder => $query->where('is_final', true)),
                Filter::make('locked')
                    ->label('Only Locked BLs')
                    ->toggle()
                    ->query(fn (Builder $query): Builder => $query->where('locked', true)),
                Tables\Filters\TrashedFilter::make(),
            ])
            ->deferFilters()
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Action::make('print_ridersheet')
                        ->label('Rider Sheet')
                        ->url(fn (?Model $record): string => route('shipping.bl.ridersheet', $record->bl))
                        ->icon('heroicon-o-newspaper')
                        // ->hidden(fn (?Model $record): string => $record->hasRiderSheet())
                        ->visible(fn (?Model $record): bool => $record->hasRiderSheet())
                        // ->color('lime')
                        ->openUrlInNewTab(),

                    Action::make('print')
                        ->url(fn (?Model $record): string => route('shipping.bl.print', $record->bl))
                        ->icon('heroicon-o-printer')
                        ->openUrlInNewTab(),
                    
                    Action::make('pdf')
                        ->label('PDF')
                        ->url(fn (?Model $record): string => route('shipping.bl.pdf', $record->bl))
                        ->icon('heroicon-o-document')
                        // ->color('success')
                        ->openUrlInNewTab(),
                
                    Tables\Actions\EditAction::make()->icon('heroicon-o-pencil-square'),
                    Tables\Actions\DeleteAction::make()->icon('heroicon-o-trash'),
                ]),
            ], position: ActionsPosition::BeforeColumns)
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBLS::route('/'),
            'create' => Pages\CreateBL::route('/create'),
            'edit' => Pages\EditBL::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
                'office'
            ]);
    }
}
