<?php

namespace App\Filament\Office\Resources;

use Filament\Forms;
use Filament\Tables;
use App\Models\Schedule;
use Filament\Forms\Form;
use Filament\Tables\Table;
use Illuminate\Support\Number;
use Filament\Infolists\Infolist;
use Filament\Resources\Resource;
use Awcodes\TableRepeater\Header;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Support\Enums\FontWeight;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\DatePicker;
use Illuminate\Database\Eloquent\Builder;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Components\ViewEntry;
use Awcodes\TableRepeater\Components\TableRepeater;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Office\Resources\ScheduleResource\Pages;
use App\Filament\Office\Resources\ScheduleResource\RelationManagers;

class ScheduleResource extends Resource
{
    protected static ?string $model = Schedule::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    protected static ?string $slug = 'shipping/schedule';
    
    protected static ?string $navigationGroup = 'Shipping';
    protected static ?string $navigationLabel = 'Schedules';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Select::make('service_route_id')
                    ->label('Service Route')
                    ->relationship('service_route', 'name')
                    ->searchable(['name'])
                    ->required(),
                Select::make('vessel_id')
                    ->label('Vessel')
                    ->relationship('vessel', 'name')
                    ->searchable(['name', 'code'])
                    ->required(),
                Select::make('operator_id')
                    ->label('Operator Company')
                    ->relationship('operator', 'name')
                    // ->required()
                    ->searchable(['name', 'code']),
                TextInput::make('voy_no')
                    ->required(),
                Toggle::make('active')
                    ->required(),
                // Select::make('status')
                //     ->required()
                //     ->options([
                //         'sailed' => 'Sailed',
                //         'arrived' => 'Arrived',
                //         'berthed' => 'Berthed',
                //         'departed' => 'Departed',
                //         ])
                //     ->default('sailed'),

                TableRepeater::make('transits')
                    ->headers([
                        Header::make('port')
                            ->width('200px')
                            ->label('Port'),
                        Header::make('eta')
                            ->label('ETA'),
                        Header::make('etb')
                            ->label('ETB'),
                        Header::make('etd')
                            ->label('ETD'),
                        Header::make('cut_off_date')
                            ->label('Cut Off Date'),
                        Header::make('string')
                            ->label('String'),
                        Header::make('reg_no')
                            ->label('Reg. No.'),
                    ])
                    ->relationship()
                    ->schema([
                        Select::make('port_of_loading_id')
                            ->relationship('port_of_loading', 'name')
                            ->searchable(['name', 'iso', 'country_name'])
                            ->required(),

                        DatePicker::make('eta')
                            ->required(),
                        DatePicker::make('etb')
                            ->required(),
                        DatePicker::make('etd')
                            ->required(),

                        DatePicker::make('cut_off_date'),
                        TextInput::make('string'),
                        TextInput::make('reg_no'),
                    ])
                    ->reorderable()
                    ->orderColumn('sort')
                    ->cloneable()
                    ->columnSpan('full'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('voy_no')
                    ->sortable(),
                Tables\Columns\TextColumn::make('vessel')
                    ->formatStateUsing(fn($state) => $state->name . ' (' . $state->code . ')')
                    // ->description(fn($state):string|null => !empty($state->capacity)?'Capacity: ' . Number::format($state->capacity).' Tues':null)
                    ->description(fn($state):string|null => !empty($state->capacity)?'Capacity: ' . Number::abbreviate($state->capacity, precision: 2).' Tues':null)
                    ->sortable(),
                Tables\Columns\TextColumn::make('operator')
                    ->formatStateUsing(fn($state) => $state->name . ' (' . $state->country->emoji . ' ' . $state->country->name . ')')
                    ->description(fn($state)=>$state?->primary_email)
                    ->sortable(),
                Tables\Columns\TextColumn::make('transits')
                    ->formatStateUsing(fn($state):string => date('d M, Y', strtotime($state->eta)) . ' ⋅ ' . $state->port_of_loading->iso)
                    ->badge(),
                Tables\Columns\TextColumn::make('service_route.name')
                    ->sortable(),
                Tables\Columns\IconColumn::make('active')
                    ->boolean(),
                // Tables\Columns\TextColumn::make('status')
                //     ->searchable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('deleted_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->modifyQueryUsing(fn (Builder $query) => $query->with(['service_route', 'operator', 'vessel', 'transits']))
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                TextEntry::make('voy_no')
                    ->label('Voyage Number')
                    ->weight(FontWeight::Bold)
                    ->size(TextEntry\TextEntrySize::Large),
                TextEntry::make('service_route.name'),
                TextEntry::make('vessel')
                    ->formatStateUsing(fn($state):string => $state->name . ' (' . $state->code . ')'),
                TextEntry::make('operator')
                    ->formatStateUsing(fn($state):string => $state->name . ' (' . $state->code . '), ' . $state->country->emoji .' '. $state->country->name),
                ViewEntry::make('transits')
                    ->view('infolists.components.transits-entry-table')
                    ->columnSpanFull(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            // RelationManagers\TransitsRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSchedules::route('/'),
            'create' => Pages\CreateSchedule::route('/create'),
            'view' => Pages\ViewSchedule::route('/{record}'),
            'edit' => Pages\EditSchedule::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
