<?php

namespace App\Filament\Office\Resources\OrganizationResource\RelationManagers;

use Filament\Forms;
use Filament\Tables;
use App\Enums\BLTypes;
use Filament\Forms\Form;
use Filament\Tables\Table;
use Filament\Tables\Actions\Action;
use Filament\Tables\Filters\Filter;
use Illuminate\Support\Facades\Auth;
use Filament\Tables\Actions\EditAction;
use Illuminate\Database\Eloquent\Model;
use Filament\Notifications\Notification;
use Filament\Tables\Actions\ActionGroup;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Actions\ReplicateAction;
use App\Filament\Office\Resources\BLResource;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Resources\RelationManagers\RelationManager;

class BlsRelationManager extends RelationManager
{
    protected static string $relationship = 'bls';
    protected static ?string $title = 'BLs';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('bl_no')
                    ->required()
                    ->maxLength(255),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('bl_no')
            ->columns([
                Tables\Columns\TextColumn::make('bl_no')
                    ->label('BL Number')
                    // ->description(fn (?Model $record): string => 'POL: '.$record->place_of_loading->name . ' (' . $record->place_of_loading->iso . ')' . ' ⭢ POD: ' . $record->port_of_discharge->name . ' (' . $record->port_of_discharge->iso . ')')
                    ->description(fn (?Model $record): string => (!empty($record->place_of_loading) && !empty($record->port_of_discharge))?($record->place_of_loading->name . ' (' . $record->place_of_loading->iso . ')' . ' → ' . $record->port_of_discharge->name . ' (' . $record->port_of_discharge->iso . ')'):'')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('etd')
                    ->label('ETD')
                    ->formatStateUsing(fn (string $state): string => \Carbon\Carbon::parse($state)->toFormattedDateString())
                    ->description(fn (?Model $record): string => 'ETA: ' . \Carbon\Carbon::parse($record->eta)->toFormattedDateString())
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('booking_ref_no')
                    ->label('Booking No.')
                    ->sortable()
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('ocean_vessel')
                    ->label('Ocean Vessel')
                    ->sortable()
                    ->searchable(isIndividual: true),
                Tables\Columns\TextColumn::make('voy_no_destination')
                    ->label('Voyage')
                    ->sortable()
                    ->searchable(isIndividual: true),
                Tables\Columns\TextColumn::make('pre_carriage_vessel')
                    ->label('Vessel')
                    ->description(fn (?Model $record): string => 'Voy: '.$record->voy_no_pre_carriage)
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('final_destination')
                    ->label('Final Destination')
                    // ->description(fn (?Model $record): string => $record->final_destination_city->name.' ⋅ '.$record->final_destination_country->name)
                    ->sortable(),
                Tables\Columns\TextColumn::make('issue_date')
                    ->label('Issued')
                    ->formatStateUsing(fn (string $state): string => \Carbon\Carbon::parse($state)->diffForHumans())
                    ->description(fn (?Model $record): string => $record->place_issue)
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\ToggleColumn::make('locked'),
                Tables\Columns\TextColumn::make('bl_containers_count')
                    ->counts('bl_containers')
                    ->formatStateUsing(fn ($state): string => $state . ($state>1?' Units':' Unit'))
                    ->description(function(?Model $record) : string {
                        return self::tues($record->bl_containers);
                    })
                    ->label('Count')
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('bl_containers')
                    ->badge()
                    ->label('Container List')
                    ->color(fn ($state): string => $state->container_size->getColor())
                    ->formatStateUsing(fn ($state): string => self::container_list($state))
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->label('Created')
                    ->description(function(?Model $record) : string|null {
                        // if (!empty($record->posted_by_user_id)) {
                        //     $model = match ($record->posted_user_type) {
                        //         1 => User::class,
                        //         2 => OfficeUser::class,
                        //     };
                        //     $user = $model::find($record->posted_by_user_id)->first();
                        //     return $user->name;
                        // }
                        // return $record->posted_by_user_id;

                        $activity = $record->activities()->where('event', 'created')->with('causer')->latest()->first();
                        $user_type = ' (Customer)';
                        if (!empty($activity->causer_type) && $activity->causer_type == 'App\Models\OfficeUser') {
                            $user_type = ' (Office)';
                        }
                        return !empty($activity->causer)?($activity->causer?->name . $user_type):null;
                    })
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->label('Updated')
                    ->description(function(?Model $record) : string|null {
                        $activity = $record->activities()->where('event', 'updated')->with('causer')->latest()->first();
                        $user_type = ' (Customer)';
                        if (!empty($activity->causer_type) && $activity->causer_type == 'App\Models\OfficeUser') {
                            $user_type = ' (Office)';
                        }
                        return !empty($activity->causer)?($activity->causer?->name . $user_type):null;
                    })
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('deleted_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->searchOnBlur()
            ->filters([
                SelectFilter::make('organization')
                    // ->relationship('organization', 'name', fn (Builder $query) => $query->withoutGlobalScopes(['office']))
                    ->relationship('organization', 'name')
                    ->multiple()
                    ->preload()
                    ->attribute('organization_id')
                    ->searchable(),
                
                SelectFilter::make('office')
                    // ->relationship('office', 'name', fn (Builder $query) => $query->withoutGlobalScopes(['office']))
                    ->relationship('office', 'name')
                    ->multiple()
                    ->preload()
                    ->attribute('office_id')
                    ->searchable()
                    ->visible(fn():bool => Auth::user()->isSuperAdmin() || Auth::user()->isAdmin()),

                Filter::make('etd')
                    ->form([
                        DatePicker::make('etd_from')
                            ->label('ETD From (date)'),
                        DatePicker::make('etd_until')
                            ->label('ETD Until (date)'),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['etd_from'],
                                fn (Builder $query, $date): Builder => $query->whereDate('etd', '>=', $date),
                            )
                            ->when(
                                $data['etd_until'],
                                fn (Builder $query, $date): Builder => $query->whereDate('etd', '<=', $date),
                            );
                    }),
                
                // Filter::make('eta')
                //     ->form([
                //         DatePicker::make('eta_from')
                //             ->label('ETA From (date)'),
                //         DatePicker::make('eta_until')
                //             ->label('ETA Until (date)'),
                //     ])
                //     ->query(function (Builder $query, array $data): Builder {
                //         return $query
                //             ->when(
                //                 $data['eta_from'],
                //                 fn (Builder $query, $date): Builder => $query->whereDate('eta', '>=', $date),
                //             )
                //             ->when(
                //                 $data['eta_until'],
                //                 fn (Builder $query, $date): Builder => $query->whereDate('eta', '<=', $date),
                //             );
                //     }),
                Tables\Filters\SelectFilter::make('bl_type')
                    ->multiple()
                    ->options(BLTypes::class),
                Tables\Filters\SelectFilter::make('bl_status')
                    ->multiple()
                    ->options(fn (): array => \App\Models\Status::query()->pluck('status', 'id')->all()),
                Filter::make('is_final')
                    ->label('Only Final BLs')
                    ->toggle()
                    ->query(fn (Builder $query): Builder => $query->where('is_final', true)),
                Filter::make('locked')
                    ->label('Only Locked BLs')
                    ->toggle()
                    ->query(fn (Builder $query): Builder => $query->where('locked', true)),
                Tables\Filters\TrashedFilter::make(),
            ])
            ->deferFilters()
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->label('Create New BL')
                    ->beforeFormFilled(function (array $data): array {
                        $data['organization_id'] = $this->getOwnerRecord()->getKey();
                        return $data;
                    })
                    ->url(route('filament.office.resources.shipping.bls.create').'?organization_id='.$this->getOwnerRecord()->getKey(), shouldOpenInNewTab: true),
            ])
            ->actions([
                ActionGroup::make([
                    Action::make('print_ridersheet')
                        ->label('Rider Sheet')
                        ->url(fn (?Model $record): string => route('shipping.bl.ridersheet', $record->bl))
                        ->icon('heroicon-o-newspaper')
                        // ->hidden(fn (?Model $record): string => $record->hasRiderSheet())
                        ->visible(fn (?Model $record): bool => $record->hasRiderSheet())
                        // ->color('lime')
                        ->openUrlInNewTab(),

                    Action::make('print_certificate')
                        ->label('Certificate')
                        ->url(fn (?Model $record): string => route('shipping.bl.certificate', $record->bl))
                        ->icon('heroicon-o-identification')
                        ->hidden(fn (?Model $record): string => $record->bl_certificates->count() < 1)
                        // ->color('lime')
                        ->openUrlInNewTab(),

                    Action::make('print')
                        ->url(fn (?Model $record): string => route('shipping.bl.show', $record->bl))
                        ->icon('heroicon-o-printer')
                        ->openUrlInNewTab(),
                    
                    Action::make('pdf')
                        ->label('PDF')
                        ->url(fn (?Model $record): string => route('shipping.bl.pdf', $record->bl))
                        ->icon('heroicon-o-document')
                        // ->color('success')
                        ->openUrlInNewTab(),

                    ReplicateAction::make()
                        ->excludeAttributes(['notify_party2', 'qty', 'discription_of_goods', 'marks', 'rider_sheet', 'weight', 'net_weight', 'measurement', 'place_issue', 'issue_date', 'shipped_on_board', 'issue_date', 'created_at','customer_created', 'bl_containers_count'])
                        // ->mutateRecordDataUsing(function (array $data): array {
                        //     $data['shipper'] = 'draft';
                        //     $data['consignee'] = 'draft';
                        //     $data['notify_party'] = 'draft';
                        //     $data['bl_no'] = 'draft';
                        //     $data['posted_user_type'] = 2;
                        //     $data['posted_by_user_id'] = auth()->id();
                     
                        //     return $data;
                        // })
                        ->beforeReplicaSaved(function (Model $replica): Model {
                            $replica->shipper = 'draft';
                            $replica->consignee = 'draft';
                            $replica->notify_party = 'draft';
                            $replica->bl_no = 'draft';
                            $replica->posted_user_type = 2;
                            $replica->posted_by_user_id = auth()->id();

                            return $replica;
                        })
                        ->successRedirectUrl(fn (Model $replica): string => route('filament.office.resources.shipping.bls.edit', [
                            'record' => $replica,
                        ]))
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('BL replicated')
                                ->body('The BL has been replicated successfully.'),
                        ),
                    Action::make('activities')
                        ->url(fn ($record) => BLResource::getUrl('activities', ['record' => $record]))
                        ->icon('heroicon-o-rectangle-stack'),
                    EditAction::make()
                        ->url(fn ($record) => BLResource::getUrl('edit', ['record' => $record]))
                        ->icon('heroicon-o-pencil-square'),
                    DeleteAction::make()->icon('heroicon-o-trash'),
                ]),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                ]),
            ])
            ->modifyQueryUsing(fn (Builder $query) => $query->withoutGlobalScopes([
                SoftDeletingScope::class,
                'organization'
            ]));
    }
}
